//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.12

import wesual.Ui 1.0

/*!
\qmltype     UiTreeLabelEdit
\inqmlmodule we.captivo.Ui
\inherits    QtQuick2::FocusScope

\brief A label editor for tree view items.
*/

FocusScope {
    id : uiLabelEdit

    /*!
    \brief Emitted when a text edit operation is finished.

    Text editing finishes when the control loses focus or the user presses
    the escape key during editing.
    */
    signal editingFinished()

    /*!
    \brief The editable text.

    \nobinding
    */
    property string text

    /*!
    \brief The placeholder text.

    The placeholder text is displayed in place of an empty text when the
    control does not have edit focus.
    */
    property string placeHolderText

    /*!
    \brief The text color of the visible text label.
    \qmlproperty color UiLabelEdit::textColor

    A label text must be able to change its text color in response to the
    state of the selected grid item. Setting this property changes the
    color of the display text only, the editing text color is unaffected.
    */
    property alias textColor : label.color

    implicitWidth    : 250
    implicitHeight   : p_.implicitHeight
    activeFocusOnTab : true

    onActiveFocusChanged : {
        if (activeFocus) {
            p_.beginEdit();
        } else {
            p_.endEdit();
        }
    }
    onTextChanged : {
        if (p_.editor) {
            p_.editor.text = text;
        }
    }

    Keys.onEscapePressed : p_.cancelEdit()

    QtObject {
        id : p_

        readonly property int horizontalMargin : 4
        readonly property int verticalMargin   : 4

        property TextInput editor : null

        readonly property int implicitHeight : {
            var iw = editor ? editor.implicitHeight : label.implicitHeight;

            iw += 2 * p_.verticalMargin;

            return iw;
        }

        function beginEdit() {
            if (editor) {
                editor.forceActiveFocus();
                editor.selectAll();
            } else {
                editor = textInput.createObject(mouseArea);
            }
        }

        function endEdit() {
            if (!editor)
                return;

            commitEdit();
            editor.destroy();
            uiLabelEdit.editingFinished();
            uiLabelEdit.focus = false;
        }

        function cancelEdit() {
            if (!editor)
                return;

            editor.destroy();
            editor = null;
            uiLabelEdit.focus = false;
        }

        function commitEdit() {
            if (!editor)
                return;

            uiLabelEdit.text = editor.text;
        }
    }

    UiTextContextMenu {
        textEditor             : p_.editor
        onContextMenuRequested : uiLabelEdit.forceActiveFocus()

    }

    MouseArea {
        id : mouseArea

        anchors.fill    : parent
        onDoubleClicked : uiLabelEdit.forceActiveFocus()
        hoverEnabled    : true
        clip            : true
        cursorShape     : uiLabelEdit.activeFocus ? Qt.IBeamCursor : Qt.ArrowCursor
    }

    Rectangle {
        anchors.fill : parent
        visible      : p_.editor !== null
        color        : UiColors.getColor(UiColors.White)
        z            : -1
        border {
            width : 1
            color : UiColors.getColor(UiColors.SelectionGreen)
        }
    }

    Text {
        id : label

        x          : p_.horizontalMargin
        y          : p_.verticalMargin
        font       : UiFonts.getFont(UiFonts.Light, 14)
        color      : UiColors.getColor(UiColors.Black)
        width      : uiLabelEdit.width  - 2 * p_.horizontalMargin
        height     : uiLabelEdit.height - 2 * p_.verticalMargin
        elide      : Text.ElideRight
        wrapMode   : Text.NoWrap
        opacity    : p_.editor ? 0 : 1
        text       : uiLabelEdit.text
        textFormat : Qt.PlainText

        horizontalAlignment : Qt.AlignLeft
    }

    Component {
        id : textInput

        TextInput {
            x      : label.x
            y      : label.y
            width  : label.width
            height : label.implicitHeight
            font   : label.font
            color  : UiColors.getColor(UiColors.Black)

            selectByMouse     : true
            selectionColor    : UiColors.getColor(UiColors.SelectionGreen)
            selectedTextColor : UiColors.getColor(UiColors.White)
            clip              : true

            Component.onCompleted : {
                text = uiLabelEdit.text;
                forceActiveFocus();
                selectAll();
            }
            Keys.onTabPressed : {
                var next = nextItemInFocusChain(true);
                if (next) {
                    next.forceActiveFocus();
                }
            }
            Keys.onEnterPressed  : p_.endEdit()
            Keys.onReturnPressed : p_.endEdit()
        }
    }
}

